<?php

namespace App\Controller\Domain\backend;

use App\Entity\Project;
use App\Entity\ProjectMedia;
use App\Form\ProjectType;
use App\Helpers\Utility;
use App\Repository\ProjectRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/project', name: 'back.project.')]
final class ProjectController extends AbstractController
{
    private string $basePath = "uploads/images/projects/";
    private string $MediaBasePath = "uploads/images/projects_medias/";

    #[Route(name: 'index', methods: ['GET'])]
    public function index(ProjectRepository $projectRepository): Response
    {
        return $this->render('domain/backend/project/index.html.twig', [
            'projects' => $projectRepository->findBy([], orderBy: ['publishedAt' => 'DESC']),
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $project = new Project();
        $form = $this->createForm(ProjectType::class, $project);
        $form->handleRequest($request);
        $files = [];

        if ($form->isSubmitted() && $form->isValid()) {
            try {
                $imageFile = $form->get('image')->getData();
                $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "projects", 1024, 683);
                if ($fileStored) {
                    $project->setImage($this->basePath . $fileStored);
                    $files[] = $this->getParameter('images_directory') . "/projects/" . basename($project->getImage());
                }

                //enregistrement des médias
                $projectMedias = $form->get('projectMedia')->getData();
                foreach ($projectMedias as $projectMedia) {
                    if ($projectMedia instanceof ProjectMedia) {
                        $mediaFile = $projectMedia->getFile();

                        $newFilename = Utility::uploadFile($mediaFile, $this->getParameter('images_directory'), "projects_medias");
                        if ($newFilename) {
                            $projectMedia->setMedia($this->MediaBasePath . $newFilename);
                            $files[] = $this->getParameter('images_directory') . "/projects_medias/" . basename($newFilename);
                        }

                        $project->addProjectMedia($projectMedia);
                        $projectMedia->setProject($project);
                        $entityManager->persist($projectMedia);
                    }
                }
                $entityManager->persist($project);
                $entityManager->flush();
            } catch (\Throwable  $exception) {
                foreach ($files as $file) {
                    if (file_exists($file)) {
                        unlink($file);
                    }
                }
                throw $exception;
            }

            return $this->redirectToRoute('back.project.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/project/new.html.twig', [
            'project' => $project,
            'form' => $form,
        ]);
    }

    #[Route('/{slug}', name: 'show', methods: ['GET'])]
    public function show($slug, ProjectRepository $projectRepository): Response
    {
        $project = $projectRepository->findOneBy(['slug' => $slug]);
        return $this->render('domain/backend/project/show.html.twig', [
            'project' => $project,
        ]);
    }

    #[Route('/{slug}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, $slug, EntityManagerInterface $entityManager, ProjectRepository $projectRepository): Response
    {
        $project = $projectRepository->findOneBy(['slug' => $slug]);
        // Sauvegardons les anciens chemins
        $oldImage = $project->getImage();
        $oldMedias = $project->getProjectMedia()->toArray();

        $form = $this->createForm(ProjectType::class, $project)
            ->add('publishedAt', DateType::class, [
                'widget' => 'single_text',
                'attr' => ['class' => 'js-datepicker'],
                'label' => 'Date du projet',
            ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            // Gestion de l'image
            $imageFile = $form->get('image')->getData();
            if ($imageFile) {
                $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "projects",1024, 683);
                if ($fileStored) {
                    // Suppression de l'ancienne image de couverture
                    $path = $this->getParameter('images_directory') . '/projects/' . basename($oldImage);
                    if ($oldImage && file_exists($path)) {
                        unlink($path);
                    }
                    $project->setImage($this->basePath . $fileStored);
                }
            }

            // Gestion des médias
            $newMedias = $form->get('projectMedia')->getData();
            $currentMediaPaths = [];

            foreach ($newMedias as $projectMedia) {
                if ($projectMedia instanceof ProjectMedia) {
                    /** @var UploadedFile $mediaFile */
                    $mediaFile = $projectMedia->getFile();
                    if ($mediaFile) {
                        $newFilename = Utility::uploadFile($mediaFile, $this->getParameter('images_directory'), "projects_medias", 1024, 683);
                        if ($newFilename) {
                            $projectMedia->setMedia($this->MediaBasePath . $newFilename);
                        }
                    }
                    $currentMediaPaths[] = $projectMedia->getMedia();
                    $project->addProjectMedia($projectMedia);
                    $projectMedia->setProject($project);
                    $entityManager->persist($projectMedia);
                }
            }

            // Suppression des anciens médias qui ne sont plus utilisés
            foreach ($oldMedias as $oldMedia) {
                if (!in_array($oldMedia->getMedia(), $currentMediaPaths)) {
                    $oldFilePath = $this->getParameter('images_directory') . '/projects_medias/' . basename($oldMedia->getMedia());
                    if (file_exists($oldFilePath)) {
                        unlink($oldFilePath);
                    }
                    $entityManager->remove($oldMedia);
                }
            }


            $entityManager->flush();

            return $this->redirectToRoute('back.project.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/project/edit.html.twig', [
            'project' => $project,
            'form' => $form,
        ]);
    }

    #[Route('/{slug}', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, $slug, EntityManagerInterface $entityManager, ProjectRepository $projectRepository): Response
    {
        $project = $projectRepository->findOneBy(['slug' => $slug]);

        if ($this->isCsrfTokenValid('delete' . $project->getSlug(), $request->getPayload()->getString('_token'))) {
            // Suppression de l'image de couverture
            $image = $project->getImage();
            if ($image) {
                $path = $this->getParameter('images_directory') . '/projects/' . basename($image);
                if (file_exists($path)) {
                    unlink($path);
                }
            }

            // Suppression des médias associés
            foreach ($project->getProjectMedia() as $media) {
                $mediaPath = $this->getParameter('images_directory') . '/projects_medias/' . basename($media->getMedia());
                if (file_exists($mediaPath)) {
                    unlink($mediaPath);
                }
            }
            $entityManager->remove($project);
            $entityManager->flush();
        }

        return $this->redirectToRoute('back.project.index', [], Response::HTTP_SEE_OTHER);
    }
}
