<?php

namespace App\Controller\Domain\backend;

use App\Entity\Station;
use App\Form\StationType;
use App\Helpers\Utility;
use App\Repository\StationRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/station', name: 'back.station.')]
final class StationController extends AbstractController
{
    private string $basePath = "uploads/images/stations/";

    #[Route(name: 'index', methods: ['GET'])]
    public function index(StationRepository $stationRepository): Response
    {
        return $this->render('domain/backend/station/index.html.twig', [
            'stations' => $stationRepository->findBy([], orderBy: ['id' => 'ASC']),
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $station = new Station();
        $form = $this->createForm(StationType::class, $station);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "stations", 410, 282);
            if ($fileStored) {
                $station->setImage($this->basePath . $fileStored);
            }
            $entityManager->persist($station);
            $entityManager->flush();

            return $this->redirectToRoute('back.station.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/station/new.html.twig', [
            'station' => $station,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Station $station): Response
    {
        return $this->render('domain/backend/station/show.html.twig', [
            'station' => $station,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Station $station, EntityManagerInterface $entityManager): Response
    {
        $oldImage = $station->getImage();
        $form = $this->createForm(StationType::class, $station);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "stations", 410, 282);
            if ($fileStored) {
                $station->setImage($this->basePath . $fileStored);

                $path = $this->getParameter('images_directory') . '/stations/' . basename($fileStored);
                if ($oldImage && file_exists($path)) {
                    unlink($path);
                }
            }
            $entityManager->flush();

            return $this->redirectToRoute('back.station.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/station/edit.html.twig', [
            'station' => $station,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, Station $station, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete' . $station->getId(), $request->getPayload()->getString('_token'))) {
            $image = $station->getImage();
            $entityManager->remove($station);
            $entityManager->flush();
            if ($image) {
                $path = $this->getParameter('images_directory') . '/slides/' . basename($image);
                if (file_exists($path)) {
                    unlink($path);
                }
            }
        }

        return $this->redirectToRoute('back.station.index', [], Response::HTTP_SEE_OTHER);
    }
}
