<?php

namespace App\Controller\Domain\frontend;

use App\Entity\Candidature;
use App\Entity\JobOffer;
use App\Form\CandidatureType;
use App\Form\ContactType;
use App\Helpers\URI;
use App\Helpers\Utility;
use App\Repository\JobOfferRepository;
use App\Repository\ParamRepository;
use App\Repository\PartnerRepository;
use App\Repository\PostRepository;
use App\Repository\ProjectRepository;
use App\Repository\ServiceRepository;
use App\Repository\SlideRepository;
use App\Repository\StationRepository;
use App\Service\StatisticsService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;
use Symfony\Component\Routing\Attribute\Route;

class HomeController extends AbstractController
{
    private $mailer;
    private StatisticsService $statisticsService;

    public function __construct(MailerInterface $mailer, StatisticsService $statisticsService)
    {
        $this->mailer = $mailer;
          $this->statisticsService = $statisticsService;

    }

    private string $basePath = "uploads/images/cv/";

    #[Route('/', name: 'front.home')]
    public function index(
        SlideRepository   $slideRepository,
        ServiceRepository $serviceRepository,
        ParamRepository   $paramRepository,
        PartnerRepository $partnerRepository,
        PostRepository    $postRepository,
        StationRepository $stationRepository,
    ): Response
    {
        $slides = $slideRepository->findAll();
        $services = $serviceRepository->findAll();
        $param = $paramRepository->findOneBy([]);
        $partners = $partnerRepository->findAll();
        $posts = $postRepository->findBy(['showHome' => true]);

        $stationsDatas = $stationRepository->findAllWithCoordinates();
        $stations = [];
        foreach ($stationsDatas as $station) {
            $stations[] = [
                'libelle' => $station->getLibelle(),
                'image' => $station->getImage() ?? "img/no-ss.png",
                'latitude' => $station->getLatitude(),
                'longitude' => $station->getLongitude(),
                'email' => $station->getEmail() ?? 'info@sifogroup.com',
                'contact' => $station->getContact() ?? '+225 0768101010',
            ];
        }

        $this->statisticsService->incrementLinkClick(URI::HOME);
        return $this->render('domain/frontend/home/index.html.twig', [
            'slides' => $slides,
            'services' => $services,
            'param' => $param,
            'partners' => $partners,
            'posts' => $posts,
            'stations' => $stations,
        ]);
    }


    #[Route('/a-propos', name: 'front.about')]
    public function about(ParamRepository $paramRepository,): Response
    {
        $param = $paramRepository->findOneBy([]);
        $this->statisticsService->incrementLinkClick(URI::ABOUT);
        return $this->render('domain/frontend/about/index.html.twig', [
            'param' => $param,
        ]);
    }

    #[Route('/notre-equipe', name: 'front.team')]
    public function team(): Response
    {
        $this->statisticsService->incrementLinkClick(URI::TEAM);
        return $this->render('domain/frontend/team/index.html.twig', []);
    }

    #[Route('/nos-produits', name: 'front.products')]
    public function product(): Response
    {
        $this->statisticsService->incrementLinkClick(URI::PRODUCT);
        return $this->render('domain/frontend/products/index.html.twig', []);
    }

    #[Route('/projets', name: 'front.projects')]
    public function gallery(ProjectRepository $projectRepository): Response
    {
        $projects = $projectRepository->findBy([], orderBy: ['publishedAt' => 'DESC']);
        $this->statisticsService->incrementLinkClick(URI::PROJECT);
        return $this->render('domain/frontend/projects/index.html.twig', [
            'projects' => $projects,
        ]);
    }

    #[Route('/station-service', name: 'front.station_service')]
    public function stationService(StationRepository $repository): Response
    {
        $stations = $repository->findBy(["published" => true]);
        $this->statisticsService->incrementLinkClick(URI::STATION_SERVICE);
        return $this->render('domain/frontend/station_service/index.html.twig', [
            'stations' => $stations,
        ]);
    }

    #[Route('/recrutement', name: 'front.recrutement')]
    public function recrutement(JobOfferRepository $jobOfferRepository): Response
    {
        $jobOffers = $jobOfferRepository->findAll();
        $this->statisticsService->incrementLinkClick(URI::RECRUTEMENT);
        return $this->render('domain/frontend/recrutement/index.html.twig', [
            'jobOffers' => $jobOffers,
        ]);
    }

    #[Route('/recrutement/{id}', name: 'front.recrutement.detail')]
    public function recrutementDetail(JobOffer $jobOffer, Request $request, EntityManagerInterface $entityManager): Response
    {
        $candidature = new Candidature();
        $form = $this->createForm(CandidatureType::class, $candidature);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Récupérer les données du formulaire
            // Gérer le fichier CV
            /** @var UploadedFile $cvFile */
            $cvFile = $form->get('cv')->getData();
            $newFilename = Utility::uploadFileNoImage($cvFile, $this->getParameter('images_directory'), "cv");
            if ($newFilename) {
                $candidature->setCv($this->basePath . $newFilename);
            }

            // Associer l'offre d'emploi à la candidature
            $candidature->setJobOffer($jobOffer);

            // Sauvegarder l'entité Candidature en base de données
            $entityManager->persist($candidature);
            $entityManager->flush();

            $this->sendMail($candidature, $jobOffer);

            // Ajouter un message de succès
            $this->addFlash('success', 'Votre candidature a été envoyée avec succès.');

            // Rediriger vers une page de confirmation ou la page d'accueil
            return $this->redirectToRoute('front.recrutement.detail', ['id' => $jobOffer->getId()]);
        }
        $this->statisticsService->incrementLinkClick(URI::RECRUTEMENT_DETAIL);
        return $this->render('domain/frontend/recrutement/details.html.twig', [
            'jobOffer' => $jobOffer,
            'form' => $form->createView(),
        ]);
    }

    #[Route('/contact', name: 'front.contact')]
    public function contact(Request $request): Response
    {
        $form = $this->createForm(ContactType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $nom = $form->get('nom')->getData();
            $email = $form->get('email')->getData();
            $message = $form->get('message')->getData();
            $objet = $form->get('objet')->getData();
            // Envoi de l'email
            $email = (new Email())
                ->from('sifogrouptic@gmail.com')
                ->to('benky47@gmail.com')
                ->subject('Confirmation de votre candidature')
                ->html($this->renderView('domain/frontend/emails/message-sending.html.twig', [
                    'nom' => $nom,
                    'email' => $email,
                    'message' => $message,
                    'objet' => $objet,
                ]));

            try {
                $this->mailer->send($email);
                $this->addFlash('success', 'Votre candidature a été envoyée avec succès.');
                return $this->redirectToRoute('front.contact');

            } catch (TransportExceptionInterface $e) {
                // Log l'erreur ou gérer l'échec de l'envoi
                $this->addFlash('error', 'L\'email de confirmation n\'a pas pu être envoyé.');
            }
        }
        $this->statisticsService->incrementLinkClick(URI::CONTACT);
        return $this->render('domain/frontend/contact/index.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    private function sendMail(Candidature $candidature, JobOffer $jobOffer)
    {
        // Envoi de l'email
        $email = (new Email())
            ->from('sifogrouptic@gmail.com')
            ->to('benky47@gmail.com')
            ->subject('Confirmation de votre candidature')
            ->html($this->renderView('domain/frontend/emails/candidature_confirmation.html.twig', [
                'candidature' => $candidature,
                'jobOffer' => $jobOffer
            ]));

        try {
            $this->mailer->send($email);

        } catch (TransportExceptionInterface $e) {
            // Log l'erreur ou gérer l'échec de l'envoi
            $this->addFlash('error', 'L\'email de confirmation n\'a pas pu être envoyé.');
        }
    }
}
